#include "GetObjects.js"
#include "GetDistance.js"

function UpdateOffsetCurveUI( factory )
{
	// Handle showing and hiding the appropriate UI depending on the offset mode.

	var mode = moi.ui.commandUI.curveoffsetmode.value;
	
	moi.ui.beginUIUpdate();
	moi.ui.hideUI( 'SelectPrompt' );
	moi.ui.hideUI( 'done' );
	moi.ui.showUI( 'CurveOptions' );
	
	if ( mode == 'throughpt' )
	{
		moi.ui.showUI( 'CurveThroughPointPrompt' );
		moi.ui.hideUI( 'CurveSidePrompt' );
	}
	else
	{
		moi.ui.showUI( 'CurveSidePrompt' );
		moi.ui.hideUI( 'CurveThroughPointPrompt' );
	}
	
	if ( !factory.AnyOpenCurves )
		moi.ui.hideUI( 'curvecapends' );
	
	moi.ui.endUIUpdate();
}

function GetLastCurveOffsetDistance()
{
	// Get the saved value for the last curve offset distance that was used, or
	// 1.0 if none were previously used.

	var dist = 1.0;
	
	try
	{
		dist = moi.command.getOption( 'OffsetCurveDistance' );
	}
	catch(e)
	{
	}
	
	return dist;
}

function SetLastCurveOffsetDistance( dist )
{
	// Save the value for the last used curve offset distance.
	moi.command.setOption( 'OffsetCurveDistance', dist );
}

function InitByDistanceMode( factory )
{
	// Initialize by distance mode - set the distance input on the factory and update
	// the distance input to that value.
	var dist = GetLastCurveOffsetDistance();
	factory.setInput( 1, dist );
	moi.ui.commandUI.curvedistanceinput.value = dist;
}

function OffsetCurves( objects )
{
	var factory = moi.command.createFactory( 'offset' );
	factory.setInput( 0, objects );

	UpdateOffsetCurveUI( factory );
	
	factory.disableUpdate( true );	

	// Bind the curvecorner select to the factory's corner input.
	moi.ui.bindUIToInput( 'curvecorner', 'value', factory.getInput(3) );
	
	// Bind the trim checkbox to the factory's trim input.
	moi.ui.bindUIToInput( 'curvetrim', 'value', factory.getInput(4) );
	
	// Bind the Both sides checkbox.
	moi.ui.bindUIToInput( 'curvebothsides', 'value', factory.getInput(6) );

	// Bind the Cap ends checkbox.
	moi.ui.bindUIToInput( 'curvecapends', 'value', factory.getInput(7) );	
	

	// If we're starting out in bydistance mode, initialize to the last used distance.
	if ( moi.ui.commandUI.curveoffsetmode.value == 'bydistance' )
		InitByDistanceMode( factory );

	factory.disableUpdate( false );

	moi.ui.commandUI.g_Factory = factory;
	
	var pointpicker = moi.ui.createPointPicker();
	
	// If there is a common plane to all the curves being offset, track points on that plane.
	var commonplane = factory.getCurvePlane();
	if ( commonplane != null )
		pointpicker.restrictToPlane( commonplane, true );

	while ( 1 )
	{
		pointpicker.clearBindings();
		pointpicker.bindResultFrame( factory.getInput(2) );

		// For through point mode, call OnCurvePoint to update the distance in the UI.
		if ( moi.ui.commandUI.curveoffsetmode.value == 'throughpt' )
			pointpicker.bindFuncPostUpdate( moi.ui.commandUI.OnCurvePoint );	
	
		if ( !pointpicker.waitForEvent() )
			return;

		if ( pointpicker.event == 'finished' )
			break;
			
		if ( pointpicker.event == 'curvedistanceinput' )
		{
			// When anything is entered into the distance input, switch to by distance mode.
			moi.ui.commandUI.curveoffsetmode.value = 'bydistance';
			
			var dist = moi.ui.commandUI.curvedistanceinput.value;
			
			factory.setInput( 1, dist );
			factory.update();
			
			SetLastCurveOffsetDistance( dist );
			UpdateOffsetCurveUI( factory );
		}
		else if ( pointpicker.event == 'curveoffsetmode' )
		{
			// The mode is being switched.
			
			// If we're switching to through point mode, then clear the distance
			// that was set on the factory. If we're switching to by distance mode
			// then do the by distance initialization.
		
			if ( moi.ui.commandUI.curveoffsetmode.value == 'throughpt' )
				factory.clearInput(1);
			else
				InitByDistanceMode( factory );
			
			factory.update();
			UpdateOffsetCurveUI( factory );
		}
	}
	
	factory.commit();
}

function OffsetBReps( objects )
{
	var factory = moi.command.createFactory( 'offset' );
	factory.setInput( 0, objects );

	// Bind the solidflip check box factory's flip input.
	factory.disableUpdate( true );
	moi.ui.bindUIToInput( 'solidflip', 'value', factory.getInput(5) );
	factory.disableUpdate( false );
	
	moi.ui.beginUIUpdate();
	moi.ui.hideUI( 'SelectPrompt' );
	moi.ui.showUI( 'SolidDistancePrompt' );
	moi.ui.showUI( 'SolidOptions' );
	moi.ui.endUIUpdate();
	
	if ( !GetDistance( factory, 1, 'soliddistanceinput' ) )
		return;
	
	factory.commit();
}

function DoOffset()
{
	var objectpicker = moi.ui.createObjectPicker();
	objectpicker.allowCurves();
	objectpicker.allowFaces();
	objectpicker.allowBReps();
	
	if ( !GetObjects( objectpicker ) )
		return;
		
	var objects = objectpicker.objects;
	
	if ( objects.numCurves > 0 )
		OffsetCurves( objects );
	else if ( objects.numBReps > 0 || objects.numFaces > 0 )
		OffsetBReps( objects );
}

DoOffset();
